<?php
// Common Functions

function sanitizeInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data);
    return $data;
}

function formatCurrency($amount) {
    return 'KSh ' . number_format($amount, 2);
}

function generateReceiptNumber() {
    return 'RCP-' . date('Ymd') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
}

function sendJsonResponse($data, $status = 200) {
    http_response_code($status);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit();
}

function logError($message, $context = []) {
    $log = date('Y-m-d H:i:s') . " - " . $message;
    if (!empty($context)) {
        $log .= " - Context: " . json_encode($context);
    }
    error_log($log . PHP_EOL, 3, 'logs/error.log');
}

function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

function validatePhone($phone) {
    // Kenyan phone number validation
    $pattern = '/^(\+254|0)[17]\d{8}$/';
    return preg_match($pattern, $phone);
}

function formatDate($date, $format = 'Y-m-d H:i:s') {
    return date($format, strtotime($date));
}

function getShopSettings($db) {
    try {
        $stmt = $db->prepare("SELECT * FROM shop_settings ORDER BY id DESC LIMIT 1");
        $stmt->execute();
        return $stmt->fetch();
    } catch (Exception $e) {
        return null;
    }
}

function getTodaysSales($db) {
    try {
        $stmt = $db->prepare("
            SELECT 
                COUNT(*) as transaction_count,
                COALESCE(SUM(total_amount), 0) as total_revenue
            FROM sales 
            WHERE DATE(created_at) = CURDATE()
        ");
        $stmt->execute();
        return $stmt->fetch();
    } catch (Exception $e) {
        return ['transaction_count' => 0, 'total_revenue' => 0];
    }
}

function getTopServices($db, $limit = 5) {
    try {
        $stmt = $db->prepare("
            SELECT 
                s.name as service_name,
                SUM(si.quantity) as total_quantity,
                SUM(si.total_price) as total_revenue
            FROM sale_items si
            JOIN services s ON si.service_id = s.id
            JOIN sales sa ON si.sale_id = sa.id
            WHERE DATE(sa.created_at) = CURDATE()
            GROUP BY s.id, s.name
            ORDER BY total_revenue DESC
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        return $stmt->fetchAll();
    } catch (Exception $e) {
        return [];
    }
}

function getCashierPerformance($db) {
    try {
        $stmt = $db->prepare("
            SELECT 
                u.full_name as cashier_name,
                COUNT(s.id) as transaction_count,
                COALESCE(SUM(s.total_amount), 0) as total_sales
            FROM sales s
            JOIN users u ON s.cashier_id = u.id
            WHERE DATE(s.created_at) = CURDATE()
            GROUP BY u.id, u.full_name
            ORDER BY total_sales DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll();
    } catch (Exception $e) {
        return [];
    }
}
?>