<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireAdmin();

$pageTitle = 'Inventory Management';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token validation failed');
    }

    $action = $_POST['action'] ?? '';

    if ($action === 'update_stock') {
        $serviceId = intval($_POST['service_id']);
        $stockQuantity = intval($_POST['stock_quantity']);
        $minStockLevel = intval($_POST['min_stock_level']);

        try {
            // Check if inventory record exists
            $stmt = $db->prepare("SELECT id FROM inventory WHERE service_id = ?");
            $stmt->execute([$serviceId]);
            $existing = $stmt->fetch();

            if ($existing) {
                $stmt = $db->prepare("
                    UPDATE inventory 
                    SET stock_quantity = ?, min_stock_level = ? 
                    WHERE service_id = ?
                ");
                $stmt->execute([$stockQuantity, $minStockLevel, $serviceId]);
            } else {
                $stmt = $db->prepare("
                    INSERT INTO inventory (service_id, stock_quantity, min_stock_level)
                    VALUES (?, ?, ?)
                ");
                $stmt->execute([$serviceId, $stockQuantity, $minStockLevel]);
            }
            $success = 'Inventory updated successfully';
        } catch (Exception $e) {
            logError("Error updating inventory: " . $e->getMessage());
            $error = 'Failed to update inventory';
        }
    }
}

// Get services with inventory
try {
    $stmt = $db->prepare("
        SELECT 
            s.*,
            c.name as category_name,
            i.stock_quantity,
            i.min_stock_level,
            CASE 
                WHEN i.stock_quantity IS NULL THEN 'No tracking'
                WHEN i.stock_quantity <= i.min_stock_level THEN 'Low stock'
                ELSE 'In stock'
            END as stock_status
        FROM services s
        LEFT JOIN categories c ON s.category_id = c.id
        LEFT JOIN inventory i ON s.id = i.service_id
        WHERE s.is_active = 1
        ORDER BY s.name
    ");
    $stmt->execute();
    $services = $stmt->fetchAll();
} catch (Exception $e) {
    logError("Error fetching inventory: " . $e->getMessage());
    $services = [];
}

include 'includes/header.php';
?>

<div class="space-y-8">
    <div>
        <h1 class="text-3xl font-bold text-gray-900">Inventory Management</h1>
        <p class="mt-2 text-gray-600">Track stock levels for physical products</p>
    </div>

    <?php if (isset($error)): ?>
    <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
        <i class="fas fa-exclamation-circle mr-2"></i>
        <?php echo htmlspecialchars($error); ?>
    </div>
    <?php endif; ?>

    <?php if (isset($success)): ?>
    <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
        <i class="fas fa-check-circle mr-2"></i>
        <?php echo htmlspecialchars($success); ?>
    </div>
    <?php endif; ?>

    <!-- Stock Alerts -->
    <?php
    $lowStockItems = array_filter($services, function($service) {
        return $service['stock_status'] === 'Low stock';
    });
    ?>

    <?php if (!empty($lowStockItems)): ?>
    <div class="bg-yellow-50 border border-yellow-200 rounded-xl p-6">
        <div class="flex items-center mb-4">
            <i class="fas fa-exclamation-triangle text-yellow-600 mr-2"></i>
            <h3 class="text-lg font-semibold text-yellow-800">Low Stock Alert</h3>
        </div>
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            <?php foreach ($lowStockItems as $item): ?>
            <div class="bg-white rounded-lg p-4 border border-yellow-200">
                <h4 class="font-medium text-gray-900"><?php echo htmlspecialchars($item['name']); ?></h4>
                <p class="text-sm text-yellow-700">
                    Stock: <?php echo $item['stock_quantity']; ?> / Min: <?php echo $item['min_stock_level']; ?>
                </p>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>

    <!-- Inventory Table -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Service/Product
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Category
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Price
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Stock
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Status
                        </th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Actions
                        </th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php foreach ($services as $service): ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div>
                                <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($service['name']); ?></div>
                                <?php if ($service['description']): ?>
                                <div class="text-sm text-gray-500"><?php echo htmlspecialchars($service['description']); ?></div>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <?php if ($service['category_name']): ?>
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                <?php echo htmlspecialchars($service['category_name']); ?>
                            </span>
                            <?php else: ?>
                            <span class="text-gray-400">No category</span>
                            <?php endif; ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                            <?php echo formatCurrency($service['unit_price']); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            <?php if ($service['stock_quantity'] !== null): ?>
                                <?php echo $service['stock_quantity']; ?> / <?php echo $service['min_stock_level']; ?>
                            <?php else: ?>
                                <span class="text-gray-400">Not tracked</span>
                            <?php endif; ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <?php
                            $statusClass = '';
                            $statusText = $service['stock_status'];
                            switch ($service['stock_status']) {
                                case 'Low stock':
                                    $statusClass = 'bg-red-100 text-red-800';
                                    break;
                                case 'In stock':
                                    $statusClass = 'bg-green-100 text-green-800';
                                    break;
                                default:
                                    $statusClass = 'bg-gray-100 text-gray-800';
                            }
                            ?>
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo $statusClass; ?>">
                                <?php echo $statusText; ?>
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                            <button
                                onclick="updateStock(<?php echo htmlspecialchars(json_encode($service)); ?>)"
                                class="text-blue-600 hover:text-blue-900 p-1 rounded"
                            >
                                <i class="fas fa-edit"></i>
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Stock Update Modal -->
<div id="stockModal" class="fixed inset-0 z-50 overflow-y-auto hidden">
    <div class="flex min-h-screen items-center justify-center p-4">
        <div class="fixed inset-0 bg-black bg-opacity-25" onclick="closeStockModal()"></div>
        
        <div class="relative bg-white rounded-xl shadow-xl max-w-md w-full p-6">
            <div class="flex items-center justify-between mb-6">
                <h3 class="text-lg font-semibold text-gray-900">Update Stock</h3>
                <button onclick="closeStockModal()" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times"></i>
                </button>
            </div>

            <form id="stockForm" method="POST">
                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">
                <input type="hidden" name="action" value="update_stock">
                <input type="hidden" name="service_id" id="stockServiceId">

                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            Service/Product
                        </label>
                        <input
                            type="text"
                            id="stockServiceName"
                            disabled
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg bg-gray-50 text-gray-500"
                        />
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            Current Stock Quantity *
                        </label>
                        <input
                            type="number"
                            name="stock_quantity"
                            id="stockQuantity"
                            required
                            min="0"
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="0"
                        />
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            Minimum Stock Level *
                        </label>
                        <input
                            type="number"
                            name="min_stock_level"
                            id="minStockLevel"
                            required
                            min="0"
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="5"
                        />
                        <p class="text-xs text-gray-500 mt-1">Alert when stock falls below this level</p>
                    </div>
                </div>

                <div class="flex space-x-3 pt-6">
                    <button
                        type="button"
                        onclick="closeStockModal()"
                        class="flex-1 bg-gray-100 text-gray-700 py-2 px-4 rounded-lg hover:bg-gray-200 transition-colors"
                    >
                        Cancel
                    </button>
                    <button
                        type="submit"
                        class="flex-1 bg-blue-600 text-white py-2 px-4 rounded-lg hover:bg-blue-700 transition-colors"
                    >
                        Update Stock
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function updateStock(service) {
    const modal = document.getElementById('stockModal');
    
    document.getElementById('stockServiceId').value = service.id;
    document.getElementById('stockServiceName').value = service.name;
    document.getElementById('stockQuantity').value = service.stock_quantity || 0;
    document.getElementById('minStockLevel').value = service.min_stock_level || 5;

    modal.classList.remove('hidden');
}

function closeStockModal() {
    document.getElementById('stockModal').classList.add('hidden');
}
</script>

</main>
</body>
</html>