<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireAdmin();

$pageTitle = 'Services Management';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token validation failed');
    }

    $action = $_POST['action'] ?? '';

    if ($action === 'add' || $action === 'edit') {
        $name = sanitizeInput($_POST['name']);
        $description = sanitizeInput($_POST['description']);
        $unitPrice = floatval($_POST['unit_price']);
        $categoryId = intval($_POST['category_id']) ?: null;
        $isActive = isset($_POST['is_active']) ? 1 : 0;

        if (empty($name) || $unitPrice <= 0) {
            $error = 'Please fill in all required fields';
        } else {
            try {
                if ($action === 'add') {
                    $stmt = $db->prepare("
                        INSERT INTO services (name, description, unit_price, category_id, is_active)
                        VALUES (?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$name, $description, $unitPrice, $categoryId, $isActive]);
                    $success = 'Service added successfully';
                } else {
                    $serviceId = intval($_POST['service_id']);
                    $stmt = $db->prepare("
                        UPDATE services 
                        SET name = ?, description = ?, unit_price = ?, category_id = ?, is_active = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([$name, $description, $unitPrice, $categoryId, $isActive, $serviceId]);
                    $success = 'Service updated successfully';
                }
            } catch (Exception $e) {
                logError("Error saving service: " . $e->getMessage());
                $error = 'Failed to save service';
            }
        }
    } elseif ($action === 'delete') {
        $serviceId = intval($_POST['service_id']);
        try {
            $stmt = $db->prepare("DELETE FROM services WHERE id = ?");
            $stmt->execute([$serviceId]);
            $success = 'Service deleted successfully';
        } catch (Exception $e) {
            logError("Error deleting service: " . $e->getMessage());
            $error = 'Failed to delete service';
        }
    }
}

// Get services and categories
try {
    $stmt = $db->prepare("
        SELECT s.*, c.name as category_name 
        FROM services s 
        LEFT JOIN categories c ON s.category_id = c.id 
        ORDER BY s.name
    ");
    $stmt->execute();
    $services = $stmt->fetchAll();

    $stmt = $db->prepare("SELECT * FROM categories ORDER BY name");
    $stmt->execute();
    $categories = $stmt->fetchAll();
} catch (Exception $e) {
    logError("Error fetching services: " . $e->getMessage());
    $services = [];
    $categories = [];
}

include 'includes/header.php';
?>

<div class="space-y-8">
    <div class="flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">Services & Products</h1>
            <p class="mt-2 text-gray-600">Manage your cyber shop services and products</p>
        </div>
        <button
            onclick="openServiceModal()"
            class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center space-x-2 transition-colors"
        >
            <i class="fas fa-plus"></i>
            <span>Add Service</span>
        </button>
    </div>

    <?php if (isset($error)): ?>
    <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
        <i class="fas fa-exclamation-circle mr-2"></i>
        <?php echo htmlspecialchars($error); ?>
    </div>
    <?php endif; ?>

    <?php if (isset($success)): ?>
    <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
        <i class="fas fa-check-circle mr-2"></i>
        <?php echo htmlspecialchars($success); ?>
    </div>
    <?php endif; ?>

    <div class="bg-white rounded-xl shadow-sm border border-gray-100">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Service
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Category
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Price
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Status
                        </th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Actions
                        </th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php foreach ($services as $service): ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div>
                                <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($service['name']); ?></div>
                                <?php if ($service['description']): ?>
                                <div class="text-sm text-gray-500"><?php echo htmlspecialchars($service['description']); ?></div>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <?php if ($service['category_name']): ?>
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                <?php echo htmlspecialchars($service['category_name']); ?>
                            </span>
                            <?php else: ?>
                            <span class="text-gray-400">No category</span>
                            <?php endif; ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                            <?php echo formatCurrency($service['unit_price']); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo $service['is_active'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                <?php echo $service['is_active'] ? 'Active' : 'Inactive'; ?>
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                            <div class="flex items-center justify-end space-x-2">
                                <button
                                    onclick="editService(<?php echo htmlspecialchars(json_encode($service)); ?>)"
                                    class="text-blue-600 hover:text-blue-900 p-1 rounded"
                                >
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button
                                    onclick="deleteService(<?php echo $service['id']; ?>, '<?php echo htmlspecialchars($service['name']); ?>')"
                                    class="text-red-600 hover:text-red-900 p-1 rounded"
                                >
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <?php if (empty($services)): ?>
        <div class="text-center py-12">
            <i class="fas fa-box text-4xl text-gray-400 mb-4"></i>
            <p class="text-gray-500">No services found</p>
            <p class="text-sm text-gray-400">Add your first service to get started</p>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Service Modal -->
<div id="serviceModal" class="fixed inset-0 z-50 overflow-y-auto hidden">
    <div class="flex min-h-screen items-center justify-center p-4">
        <div class="fixed inset-0 bg-black bg-opacity-25" onclick="closeServiceModal()"></div>
        
        <div class="relative bg-white rounded-xl shadow-xl max-w-md w-full p-6">
            <div class="flex items-center justify-between mb-6">
                <h3 id="modalTitle" class="text-lg font-semibold text-gray-900">Add New Service</h3>
                <button onclick="closeServiceModal()" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times"></i>
                </button>
            </div>

            <form id="serviceForm" method="POST">
                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">
                <input type="hidden" name="action" id="formAction" value="add">
                <input type="hidden" name="service_id" id="serviceId">

                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Service Name *
                        </label>
                        <input
                            type="text"
                            name="name"
                            id="serviceName"
                            required
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="e.g., Printing A4"
                        />
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Description
                        </label>
                        <textarea
                            name="description"
                            id="serviceDescription"
                            rows="3"
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="Optional description"
                        ></textarea>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Unit Price (KSh) *
                        </label>
                        <input
                            type="number"
                            name="unit_price"
                            id="servicePrice"
                            required
                            min="0"
                            step="0.01"
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="0.00"
                        />
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Category
                        </label>
                        <select
                            name="category_id"
                            id="serviceCategory"
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        >
                            <option value="">No category</option>
                            <?php foreach ($categories as $category): ?>
                            <option value="<?php echo $category['id']; ?>">
                                <?php echo htmlspecialchars($category['name']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="flex items-center">
                        <input
                            type="checkbox"
                            name="is_active"
                            id="serviceActive"
                            checked
                            class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                        />
                        <label for="serviceActive" class="ml-2 block text-sm text-gray-900">
                            Active
                        </label>
                    </div>
                </div>

                <div class="flex space-x-3 pt-6">
                    <button
                        type="button"
                        onclick="closeServiceModal()"
                        class="flex-1 bg-gray-100 text-gray-700 py-2 px-4 rounded-lg hover:bg-gray-200 transition-colors"
                    >
                        Cancel
                    </button>
                    <button
                        type="submit"
                        class="flex-1 bg-blue-600 text-white py-2 px-4 rounded-lg hover:bg-blue-700 transition-colors"
                    >
                        <span id="submitText">Create</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function openServiceModal(service = null) {
    const modal = document.getElementById('serviceModal');
    const form = document.getElementById('serviceForm');
    const title = document.getElementById('modalTitle');
    const submitText = document.getElementById('submitText');
    const action = document.getElementById('formAction');

    if (service) {
        // Edit mode
        title.textContent = 'Edit Service';
        submitText.textContent = 'Update';
        action.value = 'edit';
        document.getElementById('serviceId').value = service.id;
        document.getElementById('serviceName').value = service.name;
        document.getElementById('serviceDescription').value = service.description || '';
        document.getElementById('servicePrice').value = service.unit_price;
        document.getElementById('serviceCategory').value = service.category_id || '';
        document.getElementById('serviceActive').checked = service.is_active == 1;
    } else {
        // Add mode
        title.textContent = 'Add New Service';
        submitText.textContent = 'Create';
        action.value = 'add';
        form.reset();
        document.getElementById('serviceActive').checked = true;
    }

    modal.classList.remove('hidden');
}

function closeServiceModal() {
    document.getElementById('serviceModal').classList.add('hidden');
}

function editService(service) {
    openServiceModal(service);
}

function deleteService(serviceId, serviceName) {
    if (confirm(`Are you sure you want to delete "${serviceName}"?`)) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">
            <input type="hidden" name="action" value="delete">
            <input type="hidden" name="service_id" value="${serviceId}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}
</script>

</main>
</body>
</html>