<?php
require_once '../config/config.php';
require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

$auth->requireLogin();

try {
    $today = date('Y-m-d');
    
    // Today's sales summary
    $stmt = $db->prepare("
        SELECT 
            COUNT(*) as transaction_count,
            COALESCE(SUM(total_amount), 0) as total_revenue,
            COALESCE(AVG(total_amount), 0) as avg_transaction
        FROM sales 
        WHERE DATE(created_at) = ?
    ");
    $stmt->execute([$today]);
    $todayStats = $stmt->fetch();

    // This week's sales
    $stmt = $db->prepare("
        SELECT 
            COUNT(*) as transaction_count,
            COALESCE(SUM(total_amount), 0) as total_revenue
        FROM sales 
        WHERE YEARWEEK(created_at) = YEARWEEK(NOW())
    ");
    $stmt->execute();
    $weekStats = $stmt->fetch();

    // This month's sales
    $stmt = $db->prepare("
        SELECT 
            COUNT(*) as transaction_count,
            COALESCE(SUM(total_amount), 0) as total_revenue
        FROM sales 
        WHERE YEAR(created_at) = YEAR(NOW()) AND MONTH(created_at) = MONTH(NOW())
    ");
    $stmt->execute();
    $monthStats = $stmt->fetch();

    // Top services today
    $stmt = $db->prepare("
        SELECT 
            s.name as service_name,
            SUM(si.quantity) as total_quantity,
            SUM(si.total_price) as total_revenue
        FROM sale_items si
        JOIN services s ON si.service_id = s.id
        JOIN sales sa ON si.sale_id = sa.id
        WHERE DATE(sa.created_at) = ?
        GROUP BY s.id, s.name
        ORDER BY total_revenue DESC
        LIMIT 5
    ");
    $stmt->execute([$today]);
    $topServices = $stmt->fetchAll();

    // Recent transactions
    $stmt = $db->prepare("
        SELECT s.*, u.full_name as cashier_name
        FROM sales s
        JOIN users u ON s.cashier_id = u.id
        WHERE DATE(s.created_at) = ?
        ORDER BY s.created_at DESC
        LIMIT 10
    ");
    $stmt->execute([$today]);
    $recentTransactions = $stmt->fetchAll();

    // Cashier performance (admin only)
    $cashierPerformance = [];
    if ($auth->isAdmin()) {
        $stmt = $db->prepare("
            SELECT 
                u.full_name as cashier_name,
                COUNT(s.id) as transaction_count,
                COALESCE(SUM(s.total_amount), 0) as total_sales
            FROM sales s
            JOIN users u ON s.cashier_id = u.id
            WHERE DATE(s.created_at) = ?
            GROUP BY u.id, u.full_name
            ORDER BY total_sales DESC
        ");
        $stmt->execute([$today]);
        $cashierPerformance = $stmt->fetchAll();
    }

    sendJsonResponse([
        'success' => true,
        'data' => [
            'today_stats' => $todayStats,
            'week_stats' => $weekStats,
            'month_stats' => $monthStats,
            'top_services' => $topServices,
            'recent_transactions' => $recentTransactions,
            'cashier_performance' => $cashierPerformance
        ]
    ]);

} catch (Exception $e) {
    logError("Dashboard API error: " . $e->getMessage());
    sendJsonResponse(['success' => false, 'message' => 'Internal server error'], 500);
}
?>