// Common JavaScript functions for Cyber Shop POS

// Toast notification system
function showToast(message, type = 'info') {
    const toast = document.createElement('div');
    toast.className = `toast ${type}`;
    toast.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'} mr-2"></i>
        ${message}
    `;
    
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.style.animation = 'slideOut 0.3s ease-in forwards';
        setTimeout(() => {
            document.body.removeChild(toast);
        }, 300);
    }, 3000);
}

// Format currency
function formatCurrency(amount) {
    return 'KSh ' + parseFloat(amount).toLocaleString('en-KE', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}

// Confirm dialog
function confirmAction(message, callback) {
    if (confirm(message)) {
        callback();
    }
}

// Loading button state
function setButtonLoading(button, loading = true) {
    if (loading) {
        button.disabled = true;
        button.classList.add('btn-loading');
        button.dataset.originalText = button.textContent;
        button.textContent = 'Loading...';
    } else {
        button.disabled = false;
        button.classList.remove('btn-loading');
        button.textContent = button.dataset.originalText || button.textContent;
    }
}

// Form validation
function validateForm(form) {
    const requiredFields = form.querySelectorAll('[required]');
    let isValid = true;
    
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            field.classList.add('border-red-500');
            isValid = false;
        } else {
            field.classList.remove('border-red-500');
        }
    });
    
    return isValid;
}

// Search functionality
function setupSearch(searchInput, items, searchFields) {
    searchInput.addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase();
        
        items.forEach(item => {
            const searchText = searchFields.map(field => 
                item.dataset[field] || item.textContent
            ).join(' ').toLowerCase();
            
            if (searchText.includes(searchTerm)) {
                item.style.display = '';
            } else {
                item.style.display = 'none';
            }
        });
    });
}

// Auto-save functionality
function setupAutoSave(form, saveUrl, interval = 30000) {
    let autoSaveTimer;
    
    function autoSave() {
        const formData = new FormData(form);
        
        fetch(saveUrl, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast('Auto-saved', 'success');
            }
        })
        .catch(error => {
            console.error('Auto-save error:', error);
        });
    }
    
    form.addEventListener('input', function() {
        clearTimeout(autoSaveTimer);
        autoSaveTimer = setTimeout(autoSave, interval);
    });
}

// Keyboard shortcuts
function setupKeyboardShortcuts() {
    document.addEventListener('keydown', function(e) {
        // Ctrl/Cmd + S to save
        if ((e.ctrlKey || e.metaKey) && e.key === 's') {
            e.preventDefault();
            const saveButton = document.querySelector('button[type="submit"]');
            if (saveButton) {
                saveButton.click();
            }
        }
        
        // Escape to close modals
        if (e.key === 'Escape') {
            const modals = document.querySelectorAll('.modal:not(.hidden)');
            modals.forEach(modal => {
                const closeButton = modal.querySelector('[onclick*="close"]');
                if (closeButton) {
                    closeButton.click();
                }
            });
        }
    });
}

// Initialize common functionality
document.addEventListener('DOMContentLoaded', function() {
    setupKeyboardShortcuts();
    
    // Setup form validation
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            if (!validateForm(form)) {
                e.preventDefault();
                showToast('Please fill in all required fields', 'error');
            }
        });
    });
});

// Utility functions
const Utils = {
    // Debounce function
    debounce: function(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    },
    
    // Generate random ID
    generateId: function() {
        return Math.random().toString(36).substr(2, 9);
    },
    
    // Format date
    formatDate: function(date, format = 'Y-m-d H:i:s') {
        const d = new Date(date);
        const formats = {
            'Y': d.getFullYear(),
            'm': String(d.getMonth() + 1).padStart(2, '0'),
            'd': String(d.getDate()).padStart(2, '0'),
            'H': String(d.getHours()).padStart(2, '0'),
            'i': String(d.getMinutes()).padStart(2, '0'),
            's': String(d.getSeconds()).padStart(2, '0')
        };
        
        return format.replace(/[Ymdmis]/g, match => formats[match]);
    }
};