<?php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';

$auth->requireAdmin();

$pageTitle = 'Categories Management';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!$auth->validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token validation failed');
    }

    $action = $_POST['action'] ?? '';

    if ($action === 'add' || $action === 'edit') {
        $name = sanitizeInput($_POST['name']);
        $description = sanitizeInput($_POST['description']);

        if (empty($name)) {
            $error = 'Category name is required';
        } else {
            try {
                if ($action === 'add') {
                    $stmt = $db->prepare("INSERT INTO categories (name, description) VALUES (?, ?)");
                    $stmt->execute([$name, $description]);
                    $success = 'Category added successfully';
                } else {
                    $categoryId = intval($_POST['category_id']);
                    $stmt = $db->prepare("UPDATE categories SET name = ?, description = ? WHERE id = ?");
                    $stmt->execute([$name, $description, $categoryId]);
                    $success = 'Category updated successfully';
                }
            } catch (Exception $e) {
                logError("Error saving category: " . $e->getMessage());
                $error = 'Failed to save category';
            }
        }
    } elseif ($action === 'delete') {
        $categoryId = intval($_POST['category_id']);
        try {
            // Check if category has services
            $stmt = $db->prepare("SELECT COUNT(*) as count FROM services WHERE category_id = ?");
            $stmt->execute([$categoryId]);
            $result = $stmt->fetch();
            
            if ($result['count'] > 0) {
                $error = 'Cannot delete category with existing services';
            } else {
                $stmt = $db->prepare("DELETE FROM categories WHERE id = ?");
                $stmt->execute([$categoryId]);
                $success = 'Category deleted successfully';
            }
        } catch (Exception $e) {
            logError("Error deleting category: " . $e->getMessage());
            $error = 'Failed to delete category';
        }
    }
}

// Get categories
try {
    $stmt = $db->prepare("
        SELECT c.*, COUNT(s.id) as service_count 
        FROM categories c 
        LEFT JOIN services s ON c.id = s.category_id 
        GROUP BY c.id 
        ORDER BY c.name
    ");
    $stmt->execute();
    $categories = $stmt->fetchAll();
} catch (Exception $e) {
    logError("Error fetching categories: " . $e->getMessage());
    $categories = [];
}

include 'includes/header.php';
?>

<div class="space-y-8">
    <div class="flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">Categories Management</h1>
            <p class="mt-2 text-gray-600">Organize your services and products</p>
        </div>
        <button
            onclick="openCategoryModal()"
            class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center space-x-2 transition-colors"
        >
            <i class="fas fa-plus"></i>
            <span>Add Category</span>
        </button>
    </div>

    <?php if (isset($error)): ?>
    <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
        <i class="fas fa-exclamation-circle mr-2"></i>
        <?php echo htmlspecialchars($error); ?>
    </div>
    <?php endif; ?>

    <?php if (isset($success)): ?>
    <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
        <i class="fas fa-check-circle mr-2"></i>
        <?php echo htmlspecialchars($success); ?>
    </div>
    <?php endif; ?>

    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        <?php foreach ($categories as $category): ?>
        <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-lg font-semibold text-gray-900"><?php echo htmlspecialchars($category['name']); ?></h3>
                <div class="flex space-x-2">
                    <button
                        onclick="editCategory(<?php echo htmlspecialchars(json_encode($category)); ?>)"
                        class="text-blue-600 hover:text-blue-900 p-1 rounded"
                    >
                        <i class="fas fa-edit"></i>
                    </button>
                    <button
                        onclick="deleteCategory(<?php echo $category['id']; ?>, '<?php echo htmlspecialchars($category['name']); ?>')"
                        class="text-red-600 hover:text-red-900 p-1 rounded"
                    >
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
            
            <?php if ($category['description']): ?>
            <p class="text-gray-600 mb-4"><?php echo htmlspecialchars($category['description']); ?></p>
            <?php endif; ?>
            
            <div class="flex items-center text-sm text-gray-500">
                <i class="fas fa-box mr-2"></i>
                <span><?php echo $category['service_count']; ?> services</span>
            </div>
        </div>
        <?php endforeach; ?>
    </div>

    <?php if (empty($categories)): ?>
    <div class="text-center py-12">
        <i class="fas fa-tags text-4xl text-gray-400 mb-4"></i>
        <p class="text-gray-500">No categories found</p>
        <p class="text-sm text-gray-400">Add your first category to organize services</p>
    </div>
    <?php endif; ?>
</div>

<!-- Category Modal -->
<div id="categoryModal" class="fixed inset-0 z-50 overflow-y-auto hidden">
    <div class="flex min-h-screen items-center justify-center p-4">
        <div class="fixed inset-0 bg-black bg-opacity-25" onclick="closeCategoryModal()"></div>
        
        <div class="relative bg-white rounded-xl shadow-xl max-w-md w-full p-6">
            <div class="flex items-center justify-between mb-6">
                <h3 id="categoryModalTitle" class="text-lg font-semibold text-gray-900">Add New Category</h3>
                <button onclick="closeCategoryModal()" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times"></i>
                </button>
            </div>

            <form id="categoryForm" method="POST">
                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">
                <input type="hidden" name="action" id="categoryFormAction" value="add">
                <input type="hidden" name="category_id" id="categoryId">

                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Category Name *
                        </label>
                        <input
                            type="text"
                            name="name"
                            id="categoryName"
                            required
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="e.g., Printing Services"
                        />
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Description
                        </label>
                        <textarea
                            name="description"
                            id="categoryDescription"
                            rows="3"
                            class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="Optional description"
                        ></textarea>
                    </div>
                </div>

                <div class="flex space-x-3 pt-6">
                    <button
                        type="button"
                        onclick="closeCategoryModal()"
                        class="flex-1 bg-gray-100 text-gray-700 py-2 px-4 rounded-lg hover:bg-gray-200 transition-colors"
                    >
                        Cancel
                    </button>
                    <button
                        type="submit"
                        class="flex-1 bg-blue-600 text-white py-2 px-4 rounded-lg hover:bg-blue-700 transition-colors"
                    >
                        <span id="categorySubmitText">Create</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function openCategoryModal(category = null) {
    const modal = document.getElementById('categoryModal');
    const form = document.getElementById('categoryForm');
    const title = document.getElementById('categoryModalTitle');
    const submitText = document.getElementById('categorySubmitText');
    const action = document.getElementById('categoryFormAction');

    if (category) {
        // Edit mode
        title.textContent = 'Edit Category';
        submitText.textContent = 'Update';
        action.value = 'edit';
        document.getElementById('categoryId').value = category.id;
        document.getElementById('categoryName').value = category.name;
        document.getElementById('categoryDescription').value = category.description || '';
    } else {
        // Add mode
        title.textContent = 'Add New Category';
        submitText.textContent = 'Create';
        action.value = 'add';
        form.reset();
    }

    modal.classList.remove('hidden');
}

function closeCategoryModal() {
    document.getElementById('categoryModal').classList.add('hidden');
}

function editCategory(category) {
    openCategoryModal(category);
}

function deleteCategory(categoryId, categoryName) {
    if (confirm(`Are you sure you want to delete "${categoryName}"?`)) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCSRFToken(); ?>">
            <input type="hidden" name="action" value="delete">
            <input type="hidden" name="category_id" value="${categoryId}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}
</script>

</main>
</body>
</html>